import React, { useEffect, useState } from 'react';
import api, { setAuthToken } from '../services/api';
import { io } from 'socket.io-client';

function InboxPage({ auth, onLogout }) {
  const [conversations, setConversations] = useState([]);
  const [selected, setSelected] = useState(null);
  const [messages, setMessages] = useState([]);
  const [text, setText] = useState('');

  useEffect(() => {
    setAuthToken(auth.token);
    loadConversations();

    const socket = io('http://localhost:3000', {
      transports: ['websocket']
    });

    socket.on('connect', () => {
      socket.emit('join_user', auth.user.id);
    });

    socket.on('new_message', () => {
      loadConversations();
      if (selected) {
        loadMessages(selected.id);
      }
    });

    socket.on('new_conversation', () => {
      loadConversations();
    });

    return () => {
      socket.disconnect();
    };
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [auth.token]);

  async function loadConversations() {
    const { data } = await api.get('/conversations');
    setConversations(data);
  }

  async function loadMessages(conversationIdOrObj) {
    const convId = typeof conversationIdOrObj === 'object'
      ? conversationIdOrObj.id
      : conversationIdOrObj;

    const conv = conversations.find(c => c.id === convId) || selected;
    if (conv) {
      setSelected(conv);
    }
    const { data } = await api.get(`/conversations/${convId}/messages`);
    setMessages(data);
  }

  async function handleSelectConversation(conv) {
    await loadMessages(conv);
  }

  async function handleSend(e) {
    e.preventDefault();
    if (!text.trim() || !selected) return;

    const { data } = await api.post(`/conversations/${selected.id}/messages`, { body: text });
    setMessages(prev => [...prev, data]);
    setText('');
    await loadConversations();
  }

  return (
    <div style={{ display: 'flex', height: '100vh', fontFamily: 'sans-serif' }}>
      <aside style={{ width: 280, borderRight: '1px solid #ccc', padding: 8 }}>
        <div style={{ marginBottom: 12 }}>
          <strong>{auth.user.name}</strong><br />
          <button onClick={onLogout}>Sair</button>
        </div>
        <h3>Conversas</h3>
        <div style={{ overflowY: 'auto', maxHeight: '80vh' }}>
          {conversations.map(c => (
            <div
              key={c.id}
              style={{
                padding: 8,
                cursor: 'pointer',
                background: selected?.id === c.id ? '#eee' : 'transparent'
              }}
              onClick={() => handleSelectConversation(c)}
            >
              <div><strong>{c.contact_name || c.phone_number}</strong></div>
              <div style={{ fontSize: 12 }}>{c.status}</div>
            </div>
          ))}
        </div>
      </aside>

      <main style={{ flex: 1, display: 'flex', flexDirection: 'column' }}>
        {selected ? (
          <>
            <header style={{ padding: 8, borderBottom: '1px solid #ccc' }}>
              <strong>{selected.contact_name || selected.phone_number}</strong>
            </header>
            <div style={{ flex: 1, padding: 8, overflowY: 'auto' }}>
              {messages.map(m => (
                <div
                  key={m.id}
                  style={{
                    textAlign: m.direction === 'out' ? 'right' : 'left',
                    marginBottom: 6
                  }}
                >
                  <span
                    style={{
                      display: 'inline-block',
                      padding: '6px 10px',
                      borderRadius: 8,
                      background: m.direction === 'out' ? '#dcf8c6' : '#fff',
                      border: '1px solid #ccc'
                    }}
                  >
                    {m.body}
                  </span>
                </div>
              ))}
            </div>
            <form onSubmit={handleSend} style={{ padding: 8, borderTop: '1px solid #ccc' }}>
              <input
                value={text}
                onChange={e => setText(e.target.value)}
                style={{ width: '80%' }}
                placeholder="Digite uma mensagem..."
              />
              <button type="submit" style={{ marginLeft: 8 }}>Enviar</button>
            </form>
          </>
        ) : (
          <div style={{ margin: 'auto' }}>Selecione uma conversa</div>
        )}
      </main>
    </div>
  );
}

export default InboxPage;
