const router = require('express').Router();
const db = require('../config/db');
const axios = require('axios');

const WPP_API_URL = process.env.WPP_API_URL || 'http://localhost:8080';

router.get('/', async (req, res) => {
  const userId = req.user.id;

  try {
    const result = await db.query(
      `SELECT c.id,
              c.status,
              c.last_message_at,
              c.assigned_user_id,
              ct.phone_number,
              ct.name as contact_name
       FROM conversations c
       JOIN contacts ct ON ct.id = c.contact_id
       WHERE c.assigned_user_id = $1 OR c.assigned_user_id IS NULL
       ORDER BY c.last_message_at DESC NULLS LAST`,
      [userId]
    );

    res.json(result.rows);
  } catch (err) {
    console.error('Erro ao listar conversas:', err);
    res.status(500).json({ error: 'Erro interno' });
  }
});

router.get('/:id/messages', async (req, res) => {
  const { id } = req.params;

  try {
    const result = await db.query(
      `SELECT id, direction, body, created_at
       FROM messages
       WHERE conversation_id = $1
       ORDER BY created_at ASC`,
      [id]
    );

    res.json(result.rows);
  } catch (err) {
    console.error('Erro ao listar mensagens:', err);
    res.status(500).json({ error: 'Erro interno' });
  }
});

router.post('/:id/messages', async (req, res) => {
  const { id } = req.params;
  const { body } = req.body;
  const userId = req.user.id;
  const io = req.app.get('io');

  try {
    const convRes = await db.query(
      `SELECT c.*, ct.phone_number, ws.name as session_name
       FROM conversations c
       JOIN contacts ct ON ct.id = c.contact_id
       JOIN wa_sessions ws ON ws.id = c.wa_session_id
       WHERE c.id = $1`,
      [id]
    );

    if (!convRes.rowCount) {
      return res.status(404).json({ error: 'Conversa não encontrada' });
    }

    const conversation = convRes.rows[0];

    if (conversation.assigned_user_id && conversation.assigned_user_id !== userId) {
      return res.status(403).json({ error: 'Você não é o dono desta conversa' });
    }

    if (!conversation.assigned_user_id) {
      await db.query(
        'UPDATE conversations SET assigned_user_id = $1 WHERE id = $2',
        [userId, id]
      );
      conversation.assigned_user_id = userId;
    }

    await axios.post(
      `${WPP_API_URL}/api/${conversation.session_name}/send-message`,
      {
        phone: conversation.phone_number,
        message: body
      }
    );

    const msgRes = await db.query(
      `INSERT INTO messages (conversation_id, direction, body)
       VALUES ($1, 'out', $2)
       RETURNING *`,
      [conversation.id, body]
    );
    const message = msgRes.rows[0];

    await db.query(
      'UPDATE conversations SET last_message_at = NOW() WHERE id = $1',
      [conversation.id]
    );

    io.to(`conversation:${conversation.id}`).emit('message_sent', { conversation, message });

    res.json(message);
  } catch (err) {
    console.error('Erro ao enviar mensagem:', err.response?.data || err);
    res.status(500).json({ error: 'Erro ao enviar mensagem' });
  }
});

module.exports = router;
