<?php
// plataformas_api.php
// CRUD simples das plataformas do usuário via AJAX.

require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

header('Content-Type: application/json; charset=utf-8');

$usuarioEmail = $_SESSION['usuario_email'] ?? null;
if (!$usuarioEmail) {
    http_response_code(401);
    echo json_encode(['ok' => false, 'msg' => 'Usuário não autenticado.']);
    exit;
}

$acao = $_POST['acao'] ?? $_GET['acao'] ?? '';

try {
    if ($acao === 'listar') {
        $plataformas = load_plataformas($pdo, $usuarioEmail);
        echo json_encode([
            'ok'          => true,
            'plataformas' => $plataformas,
        ]);
        exit;
    }

    if ($acao === 'salvar') {
        $nome       = trim((string)($_POST['nome'] ?? ''));
        $slug       = trim((string)($_POST['slug'] ?? ''));
        $taxa       = (float)($_POST['taxa_percent'] ?? 0);
        $custoFixo  = (float)($_POST['custo_fixo'] ?? 0);
        $origSlug   = trim((string)($_POST['original_slug'] ?? ''));

        if ($nome === '') {
            throw new Exception('Nome da plataforma é obrigatório.');
        }

        // Se não veio slug, gera um slug básico a partir do nome
        if ($slug === '') {
            $slug = preg_replace('/[^a-z0-9]+/i', '_', strtolower(iconv('UTF-8', 'ASCII//TRANSLIT', $nome)));
            $slug = trim($slug, '_');
        }

        // Busca dados ANTES (para log)
        $antes = null;
        $slugConsulta = $origSlug !== '' ? $origSlug : $slug;
        $stmtAntes = $pdo->prepare("SELECT slug, nome, taxa_percent, custo_fixo
                                      FROM plataformas
                                     WHERE user_email = :email
                                       AND slug = :slug
                                     LIMIT 1");
        $stmtAntes->execute([
            ':email' => $usuarioEmail,
            ':slug'  => $slugConsulta,
        ]);
        $rowAntes = $stmtAntes->fetch(PDO::FETCH_ASSOC);
        if ($rowAntes) {
            $antes = [
                'slug'        => $rowAntes['slug'],
                'nome'        => $rowAntes['nome'],
                'taxa_percent'=> (float)$rowAntes['taxa_percent'],
                'custo_fixo'  => (float)$rowAntes['custo_fixo'],
            ];
        }

        // Salva plataforma (função existente no functions.php)
        salvar_plataforma($pdo, $usuarioEmail, $slug, $nome, $taxa, $custoFixo, $origSlug);

        // Busca dados DEPOIS (para log)
        $stmtDepois = $pdo->prepare("SELECT slug, nome, taxa_percent, custo_fixo
                                       FROM plataformas
                                      WHERE user_email = :email
                                        AND slug = :slug
                                      LIMIT 1");
        $stmtDepois->execute([
            ':email' => $usuarioEmail,
            ':slug'  => $slug,
        ]);
        $rowDepois = $stmtDepois->fetch(PDO::FETCH_ASSOC);
        $depois = $rowDepois ? [
            'slug'        => $rowDepois['slug'],
            'nome'        => $rowDepois['nome'],
            'taxa_percent'=> (float)$rowDepois['taxa_percent'],
            'custo_fixo'  => (float)$rowDepois['custo_fixo'],
        ] : null;

        $acaoLog = $antes ? 'update' : 'create';
        registrar_log_auditoria(
            $pdo,
            'usuario',
            $usuarioEmail,
            'plataforma',
            $usuarioEmail,
            $acaoLog,
            $slug,
            $antes,
            $depois
        );

        echo json_encode(['ok' => true]);
        exit;
    }

    if ($acao === 'excluir') {
        $slug = trim((string)($_POST['slug'] ?? ''));

        if ($slug === '') {
            throw new Exception('Slug da plataforma não informado.');
        }

        // Busca dados ANTES (para log)
        $stmtAntes = $pdo->prepare("SELECT slug, nome, taxa_percent, custo_fixo
                                      FROM plataformas
                                     WHERE user_email = :email
                                       AND slug = :slug
                                     LIMIT 1");
        $stmtAntes->execute([
            ':email' => $usuarioEmail,
            ':slug'  => $slug,
        ]);
        $rowAntes = $stmtAntes->fetch(PDO::FETCH_ASSOC);
        $antes = $rowAntes ? [
            'slug'        => $rowAntes['slug'],
            'nome'        => $rowAntes['nome'],
            'taxa_percent'=> (float)$rowAntes['taxa_percent'],
            'custo_fixo'  => (float)$rowAntes['custo_fixo'],
        ] : null;

        excluir_plataforma($pdo, $usuarioEmail, $slug);

        registrar_log_auditoria(
            $pdo,
            'usuario',
            $usuarioEmail,
            'plataforma',
            $usuarioEmail,
            'delete',
            $slug,
            $antes,
            null
        );

        echo json_encode(['ok' => true]);
        exit;
    }

    throw new Exception('Ação inválida.');

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'ok'  => false,
        'msg' => $e->getMessage(),
    ]);
    exit;
}
