<?php
// historico.php
// Dashboard simples com histórico de simulações do usuário logado.

require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (empty($_SESSION['usuario_autorizado']) || empty($_SESSION['usuario_email'])) {
    header('Location: login.php');
    exit;
}

$usuarioEmail = $_SESSION['usuario_email'];
$pageTitle = 'Histórico de Simulações';

$limite = 100;
try {
    $stmt = $pdo->prepare("SELECT *
                             FROM simulacoes
                            WHERE user_email = :email
                            ORDER BY criado_em DESC
                            LIMIT :limite");
    $stmt->bindValue(':email', $usuarioEmail, PDO::PARAM_STR);
    $stmt->bindValue(':limite', (int)$limite, PDO::PARAM_INT);
    $stmt->execute();
    $simulacoes = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $simulacoes = [];
}

include __DIR__ . '/header.php';
?>
<div class="page">
  <div class="content">
    <div class="card">
      <div class="card-header">
        <div class="card-title">Histórico de simulações</div>
        <span class="badge"><?php echo count($simulacoes); ?> registros</span>
      </div>

      <div class="field">
        <p class="hint">
          Aqui você vê as últimas simulações realizadas com sua conta.
          Você pode exportar em PDF ou Excel pelos botões abaixo.
        </p>
      </div>

      <div class="field" style="display:flex;gap:8px;flex-wrap:wrap;">
        <a class="btn" style="text-decoration:none;" href="export_simulacoes.php?formato=pdf" target="_blank">
          <span class="dot"></span> Exportar PDF
        </a>
        <a class="btn" style="text-decoration:none;" href="export_simulacoes.php?formato=excel" target="_blank">
          <span class="dot"></span> Exportar Excel
        </a>
      </div>

      <div class="field" style="overflow:auto; max-height:480px;">
        <?php if (empty($simulacoes)): ?>
          <p style="font-size:0.85rem;color:var(--muted);">
            Nenhuma simulação registrada ainda. Use a tela principal do APP para fazer seus cálculos.
          </p>
        <?php else: ?>
          <table style="width:100%;border-collapse:collapse;font-size:0.78rem;">
            <thead>
              <tr>
                <th style="text-align:left;padding:6px;border-bottom:1px solid var(--border-subtle);">Data</th>
                <th style="text-align:left;padding:6px;border-bottom:1px solid var(--border-subtle);">Plataforma</th>
                <th style="text-align:right;padding:6px;border-bottom:1px solid var(--border-subtle);">Preço custo</th>
                <th style="text-align:right;padding:6px;border-bottom:1px solid var(--border-subtle);">Preço venda</th>
                <th style="text-align:right;padding:6px;border-bottom:1px solid var(--border-subtle);">Lucro (R$)</th>
                <th style="text-align:right;padding:6px;border-bottom:1px solid var(--border-subtle);">Lucro (%)</th>
                <th style="text-align:right;padding:6px;border-bottom:1px solid var(--border-subtle);">Impostos (%)</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($simulacoes as $row): ?>
                <tr>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);">
                    <?php
                      $dt = $row['criado_em'] ?? '';
                      if ($dt) {
                          $ts = strtotime($dt);
                          echo $ts ? date('d/m/Y H:i', $ts) : htmlspecialchars($dt);
                      }
                    ?>
                  </td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);">
                    <?php echo htmlspecialchars($row['plataforma_slug'] ?: '-'); ?>
                  </td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);text-align:right;">
                    R$ <?php echo number_format((float)$row['preco_custo'], 2, ',', '.'); ?>
                  </td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);text-align:right;">
                    R$ <?php echo number_format((float)$row['preco_venda'], 2, ',', '.'); ?>
                  </td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);text-align:right;">
                    R$ <?php echo number_format((float)$row['lucro_valor'], 2, ',', '.'); ?>
                  </td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);text-align:right;">
                    <?php echo number_format((float)$row['lucro_percent'] * 100, 2, ',', '.'); ?>%
                  </td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);text-align:right;">
                    <?php echo number_format((float)$row['impostos_percent'] * 100, 2, ',', '.'); ?>%
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        <?php endif; ?>
      </div>
    </div>
  </div>
</div>

<?php include __DIR__ . '/footer.php'; ?>
