<?php
// functions.php
// Funções auxiliares para configs e gateways

function get_app_setting(PDO $pdo, string $key, $default = null): ?string {
    $stmt = $pdo->prepare("SELECT `value` FROM app_settings WHERE `key` = :key LIMIT 1");
    $stmt->execute([':key' => $key]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);
    if ($row && $row['value'] !== null) {
        return $row['value'];
    }
    return $default;
}

function set_app_setting(PDO $pdo, string $key, $value): void {
    $stmt = $pdo->prepare("
        INSERT INTO app_settings (`key`, `value`)
        VALUES (:key, :value)
        ON DUPLICATE KEY UPDATE `value` = VALUES(`value`)
    ");
    $stmt->execute([':key' => $key, ':value' => $value]);
}

function get_all_app_settings(PDO $pdo): array {
    $stmt = $pdo->query("SELECT `key`, `value` FROM app_settings");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    $out = [];
    foreach ($rows as $r) {
        $out[$r['key']] = $r['value'];
    }
    return $out;
}

function hydrate_gateway_row(array $gw): array {
    $gw['config'] = [];
    if (!empty($gw['config_json'])) {
        $cfg = json_decode($gw['config_json'], true);
        if (is_array($cfg)) {
            $gw['config'] = $cfg;
        }
    }
    return $gw;
}

function get_gateways(PDO $pdo): array {
    $stmt = $pdo->query("SELECT * FROM payment_gateways ORDER BY id ASC");
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
    foreach ($rows as &$r) {
        $r = hydrate_gateway_row($r);
    }
    return $rows;
}

function get_gateway_by_id(PDO $pdo, int $id): ?array {
    $stmt = $pdo->prepare("SELECT * FROM payment_gateways WHERE id = :id LIMIT 1");
    $stmt->execute([':id' => $id]);
    $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$gw) return null;
    return hydrate_gateway_row($gw);
}

function get_gateway_by_slug(PDO $pdo, string $slug): ?array {
    $stmt = $pdo->prepare("SELECT * FROM payment_gateways WHERE slug = :slug LIMIT 1");
    $stmt->execute([':slug' => $slug]);
    $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$gw) return null;
    return hydrate_gateway_row($gw);
}

function get_default_gateway(PDO $pdo): ?array {
    $stmt = $pdo->query("SELECT * FROM payment_gateways WHERE is_default = 1 AND is_active = 1 LIMIT 1");
    $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$gw) {
        $stmt = $pdo->query("SELECT * FROM payment_gateways WHERE is_active = 1 LIMIT 1");
        $gw = $stmt->fetch(PDO::FETCH_ASSOC);
    }
    if (!$gw) return null;
    return hydrate_gateway_row($gw);
}

function set_default_gateway(PDO $pdo, int $id): void {
    $pdo->beginTransaction();
    try {
        $pdo->exec("UPDATE payment_gateways SET is_default = 0");
        $stmt = $pdo->prepare("UPDATE payment_gateways SET is_default = 1 WHERE id = :id");
        $stmt->execute([':id' => $id]);
        $pdo->commit();
    } catch (Throwable $e) {
        $pdo->rollBack();
        throw $e;
    }
}

// Verifica se o e-mail tem ao menos um pagamento aprovado em qualquer gateway
function usuario_tem_pagamento_aprovado(PDO $pdo, string $email): bool {
    $stmt = $pdo->prepare("
        SELECT 1
        FROM vendas_app
        WHERE email = :email
          AND status = 'approved'
        LIMIT 1
    ");
    $stmt->execute([':email' => $email]);
    return (bool)$stmt->fetchColumn();
}

// ----------------------------
// Plataformas (CRUD)
// ----------------------------

function load_plataformas(PDO $pdo, string $userEmail): array {
    $stmt = $pdo->prepare("
        SELECT slug, nome, taxa_percent, custo_fixo
        FROM plataformas
        WHERE user_email = :email
        ORDER BY nome
    ");
    $stmt->execute([':email' => $userEmail]);
    $rows = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $out = [];
    foreach ($rows as $r) {
        $out[$r['slug']] = [
            'nome'        => $r['nome'],
            'taxaPercent' => (float) $r['taxa_percent'],
            'custoFixo'   => (float) $r['custo_fixo'],
        ];
    }
    return $out;
}

function gerar_slug_plataforma(string $nome): string {
    $slug = mb_strtolower($nome, 'UTF-8');
    $slug = iconv('UTF-8', 'ASCII//TRANSLIT', $slug);
    $slug = preg_replace('/[^a-z0-9]+/', '_', $slug);
    $slug = preg_replace('/_+/', '_', $slug);
    $slug = trim($slug, '_');
    if ($slug === '') {
        $slug = 'plataforma_' . uniqid();
    }
    return $slug;
}

function salvar_plataforma(
    PDO $pdo,
    string $userEmail,
    string $slug,
    string $nome,
    float $taxaPercent,
    float $custoFixo,
    ?string $originalSlug = null
): void {

    if ($originalSlug) {
        // UPDATE (edição)
        $stmt = $pdo->prepare("
            UPDATE plataformas
               SET slug         = :slug,
                   nome         = :nome,
                   taxa_percent = :taxa,
                   custo_fixo   = :custo
             WHERE user_email = :email
               AND slug       = :original
            LIMIT 1
        ");
        $stmt->execute([
            ':slug'     => $slug,
            ':nome'     => $nome,
            ':taxa'     => $taxaPercent,
            ':custo'    => $custoFixo,
            ':email'    => $userEmail,
            ':original' => $originalSlug,
        ]);
    } else {
        // INSERT (nova)
        $stmt = $pdo->prepare("
            INSERT INTO plataformas (user_email, slug, nome, taxa_percent, custo_fixo)
            VALUES (:email, :slug, :nome, :taxa, :custo)
            ON DUPLICATE KEY UPDATE
              nome         = VALUES(nome),
              taxa_percent = VALUES(taxa_percent),
              custo_fixo   = VALUES(custo_fixo)
        ");
        $stmt->execute([
            ':email' => $userEmail,
            ':slug'  => $slug,
            ':nome'  => $nome,
            ':taxa'  => $taxaPercent,
            ':custo' => $custoFixo,
        ]);
    }
}

function excluir_plataforma(PDO $pdo, string $userEmail, string $slug): void {
    $stmt = $pdo->prepare("
        DELETE FROM plataformas
         WHERE user_email = :email
           AND slug       = :slug
        LIMIT 1
    ");
    $stmt->execute([
        ':email' => $userEmail,
        ':slug'  => $slug,
    ]);
}

function salvar_user_defaults(PDO $pdo, string $userEmail, array $dados): void {
    $stmt = $pdo->prepare("
        INSERT INTO user_defaults (
            user_email,
            cmv,
            despesas_fixas,
            taxa_mentoria,
            custo_fixo_plataforma,
            taxa_plataforma,
            fat_12m,
            lucro_desejado
        ) VALUES (
            :email,
            :cmv,
            :despesas,
            :taxa_mentoria,
            :custo_fixo,
            :taxa_plataforma,
            :fat_12m,
            :lucro
        )
        ON DUPLICATE KEY UPDATE
          cmv                   = VALUES(cmv),
          despesas_fixas        = VALUES(despesas_fixas),
          taxa_mentoria         = VALUES(taxa_mentoria),
          custo_fixo_plataforma = VALUES(custo_fixo_plataforma),
          taxa_plataforma       = VALUES(taxa_plataforma),
          fat_12m               = VALUES(fat_12m),
          lucro_desejado        = VALUES(lucro_desejado)
    ");

    $stmt->execute([
        ':email'           => $userEmail,
        ':cmv'             => (float)$dados['cmv'],
        ':despesas'        => (float)$dados['despesas_fixas'],
        ':taxa_mentoria'   => (float)$dados['taxa_mentoria'],
        ':custo_fixo'      => (float)$dados['custo_fixo_plataforma'],
        ':taxa_plataforma' => (float)$dados['taxa_plataforma'],
        ':fat_12m'         => (float)$dados['fat_12m'],
        ':lucro'           => (float)$dados['lucro_desejado'],
    ]);
}

function get_user_defaults(PDO $pdo, string $userEmail, array $defaultsGlobais): array {
    $stmt = $pdo->prepare("
        SELECT
          cmv,
          despesas_fixas,
          taxa_mentoria,
          custo_fixo_plataforma,
          taxa_plataforma,
          fat_12m,
          lucro_desejado
        FROM user_defaults
        WHERE user_email = :email
        LIMIT 1
    ");
    $stmt->execute([':email' => $userEmail]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        return $defaultsGlobais;
    }

    return [
        'cmv'                   => number_format((float)$row['cmv'], 2, ',', ''),
        'despesas_fixas'        => number_format((float)$row['despesas_fixas'], 2, ',', ''),
        'taxa_mentoria'         => number_format((float)$row['taxa_mentoria'], 2, ',', ''),
        'custo_fixo_plataforma' => number_format((float)$row['custo_fixo_plataforma'], 2, ',', ''),
        'taxa_plataforma'       => number_format((float)$row['taxa_plataforma'], 2, ',', ''),
        'fat_12m'               => number_format((float)$row['fat_12m'], 2, ',', ''),
        'lucro_desejado'        => number_format((float)$row['lucro_desejado'], 2, ',', ''),
    ];
}





// ----------------------------
// Logs de auditoria (plataformas, padrões, etc.)
// ----------------------------
/**
 * Registra um log simples de auditoria.
 *
 * @param PDO    $pdo
 * @param string $actorTipo       'admin' ou 'usuario'
 * @param string|null $actorEmail E-mail de quem fez a alteração
 * @param string $alvoTipo        'plataforma' ou 'defaults'
 * @param string|null $alvoUserEmail E-mail do usuário dono dos dados alterados
 * @param string $acao            'create', 'update' ou 'delete'
 * @param string|null $referencia Slug da plataforma ou outra referência
 * @param array|null  $dadosAntes Dados antes da alteração (array associativo)
 * @param array|null  $dadosDepois Dados depois da alteração (array associativo)
 */
function registrar_log_auditoria(
    PDO $pdo,
    string $actorTipo,
    ?string $actorEmail,
    string $alvoTipo,
    ?string $alvoUserEmail,
    string $acao,
    ?string $referencia,
    ?array $dadosAntes = null,
    ?array $dadosDepois = null
): void {
    try {
        $sql = "INSERT INTO logs_auditoria (
                    criado_em,
                    actor_tipo,
                    actor_email,
                    alvo_tipo,
                    alvo_user_email,
                    acao,
                    referencia,
                    dados_antes,
                    dados_depois
                ) VALUES (
                    NOW(),
                    :actor_tipo,
                    :actor_email,
                    :alvo_tipo,
                    :alvo_user_email,
                    :acao,
                    :referencia,
                    :dados_antes,
                    :dados_depois
                )";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            ':actor_tipo'      => $actorTipo,
            ':actor_email'     => $actorEmail,
            ':alvo_tipo'       => $alvoTipo,
            ':alvo_user_email' => $alvoUserEmail,
            ':acao'            => $acao,
            ':referencia'      => $referencia,
            ':dados_antes'     => $dadosAntes  ? json_encode($dadosAntes, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : null,
            ':dados_depois'    => $dadosDepois ? json_encode($dadosDepois, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) : null,
        ]);
    } catch (Throwable $e) {
        // Não interrompe o fluxo principal por causa de falha no log.
        // Em produção você poderia logar isso em arquivo.
    }
}

