<?php
// export_simulacoes.php
// Exporta o histórico de simulações do usuário logado em PDF ou Excel (CSV).

require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (empty($_SESSION['usuario_autorizado']) || empty($_SESSION['usuario_email'])) {
    header('Location: login.php');
    exit;
}

$usuarioEmail = $_SESSION['usuario_email'];
$formato = strtolower($_GET['formato'] ?? 'excel');

$limite = 1000;

try {
    $stmt = $pdo->prepare("SELECT *
                             FROM simulacoes
                            WHERE user_email = :email
                            ORDER BY criado_em DESC
                            LIMIT :limite");
    $stmt->bindValue(':email', $usuarioEmail, PDO::PARAM_STR);
    $stmt->bindValue(':limite', (int)$limite, PDO::PARAM_INT);
    $stmt->execute();
    $simulacoes = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $simulacoes = [];
}

if ($formato === 'pdf') {
    require __DIR__ . '/vendor/autoload.php';
    try {
        $html = '<h2 style="font-family:sans-serif;">Histórico de simulações</h2>';
        if (empty($simulacoes)) {
            $html .= '<p style="font-family:sans-serif;font-size:12px;">Nenhuma simulação encontrada.</p>';
        } else {
            $html .= '<table style="width:100%;border-collapse:collapse;font-size:11px;font-family:sans-serif;">';
            $html .= '<thead><tr>';
            $cols = ['Data', 'Plataforma', 'Preço custo', 'Preço venda', 'Lucro (R$)', 'Lucro (%)', 'Impostos (%)'];
            foreach ($cols as $c) {
                $html .= '<th style="border-bottom:1px solid #ccc;text-align:left;padding:4px 6px;">' . htmlspecialchars($c) . '</th>';
            }
            $html .= '</tr></thead><tbody>';
            foreach ($simulacoes as $row) {
                $dt = $row['criado_em'] ?? '';
                if ($dt) {
                    $ts = strtotime($dt);
                    $dtFmt = $ts ? date('d/m/Y H:i', $ts) : $dt;
                } else {
                    $dtFmt = '';
                }
                $plataforma = $row['plataforma_slug'] ?: '-';
                $precoCusto = number_format((float)$row['preco_custo'], 2, ',', '.');
                $precoVenda = number_format((float)$row['preco_venda'], 2, ',', '.');
                $lucroValor = number_format((float)$row['lucro_valor'], 2, ',', '.');
                $lucroPercent = number_format((float)$row['lucro_percent'] * 100, 2, ',', '.');
                $impPercent = number_format((float)$row['impostos_percent'] * 100, 2, ',', '.');

                $html .= '<tr>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">' . htmlspecialchars($dtFmt) . '</td>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">' . htmlspecialchars($plataforma) . '</td>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">R$ ' . $precoCusto . '</td>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">R$ ' . $precoVenda . '</td>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">R$ ' . $lucroValor . '</td>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">' . $lucroPercent . '%</td>';
                $html .= '<td style="border-bottom:1px solid #eee;padding:3px 6px;">' . $impPercent . '%</td>';
                $html .= '</tr>';
            }
            $html .= '</tbody></table>';
        }

        $dompdf = new Dompdf\Dompdf();
        $dompdf->loadHtml($html, 'UTF-8');
        $dompdf->setPaper('A4', 'landscape');
        $dompdf->render();

        $dompdf->stream('simulacoes.pdf', ['Attachment' => true]);
        exit;
    } catch (Throwable $e) {
        // Em caso de erro no PDF, cai para CSV/Excel
        $formato = 'excel';
    }
}

// Export padrão: CSV (compatível com Excel)
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="simulacoes.csv"');

$out = fopen('php://output', 'w');

// Cabeçalho
fputcsv($out, [
    'Data',
    'Plataforma',
    'Preço_custo',
    'Preço_venda',
    'Lucro_reais',
    'Lucro_percentual',
    'Impostos_percentual',
    'CMV',
    'Custos_operacionais',
    'Custos_fixos_plataforma',
    'Comissao_gestao',
    'Comissao_plataforma',
    'Lucro_desejado',
    'Faturamento_12m',
    'Regime_tributario'
], ';');

// Linhas
foreach ($simulacoes as $row) {
    $dt = $row['criado_em'] ?? '';
    if ($dt) {
        $ts = strtotime($dt);
        $dtFmt = $ts ? date('d/m/Y H:i', $ts) : $dt;
    } else {
        $dtFmt = '';
    }

    fputcsv($out, [
        $dtFmt,
        $row['plataforma_slug'] ?: '-',
        number_format((float)$row['preco_custo'], 2, ',', '.'),
        number_format((float)$row['preco_venda'], 2, ',', '.'),
        number_format((float)$row['lucro_valor'], 2, ',', '.'),
        number_format((float)$row['lucro_percent'] * 100, 2, ',', '.'),
        number_format((float)$row['impostos_percent'] * 100, 2, ',', '.'),
        number_format((float)$row['cmv'], 2, ',', '.'),
        number_format((float)$row['custos_operacionais'], 2, ',', '.'),
        number_format((float)$row['custos_fixos_plataforma'], 2, ',', '.'),
        number_format((float)$row['comissao_gestao'] * 100, 2, ',', '.'),
        number_format((float)$row['comissao_plataforma'] * 100, 2, ',', '.'),
        number_format((float)$row['lucro_desejado'] * 100, 2, ',', '.'),
        number_format((float)$row['fat_12m'], 2, ',', '.'),
        $row['regime_tributario'] ?: ''
    ], ';');
}

fclose($out);
exit;
