<?php
// admin_logs.php
// Listagem simples de logs de auditoria para o painel Admin.

require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (empty($_SESSION['admin_autorizado'])) {
    header('Location: admin_login.php');
    exit;
}

$pageTitle = 'Logs de Auditoria - APP Formação de Preço';

try {
    $stmt = $pdo->query("SELECT *
                           FROM logs_auditoria
                          ORDER BY criado_em DESC
                          LIMIT 300");
    $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
    $logs = [];
}

include __DIR__ . '/header.php';
?>
<div class="page">
  <div class="content">
    <div class="card">
      <div class="card-header">
        <div class="card-title">Logs de auditoria</div>
        <span class="badge"><?php echo count($logs); ?> eventos</span>
      </div>

      <div class="field">
        <p class="hint">
          Aqui você acompanha alterações feitas nas plataformas e nos padrões salvos pelos usuários.
        </p>
      </div>

      <div class="field" style="overflow:auto; max-height:520px;">
        <?php if (empty($logs)): ?>
          <p style="font-size:0.85rem;color:var(--muted);">
            Nenhum log registrado ainda.
          </p>
        <?php else: ?>
          <table style="width:100%;border-collapse:collapse;font-size:0.78rem;">
            <thead>
              <tr>
                <th style="padding:6px;border-bottom:1px solid var(--border-subtle);text-align:left;">Data</th>
                <th style="padding:6px;border-bottom:1px solid var(--border-subtle);text-align:left;">Tipo</th>
                <th style="padding:6px;border-bottom:1px solid var(--border-subtle);text-align:left;">Usuário alvo</th>
                <th style="padding:6px;border-bottom:1px solid var(--border-subtle);text-align:left;">Ação</th>
                <th style="padding:6px;border-bottom:1px solid var(--border-subtle);text-align:left;">Referência</th>
                <th style="padding:6px;border-bottom:1px solid var(--border-subtle);text-align:left;">Resumo</th>
              </tr>
            </thead>
            <tbody>
              <?php foreach ($logs as $log): ?>
                <?php
                  $dt = $log['criado_em'] ?? '';
                  if ($dt) {
                      $ts = strtotime($dt);
                      $dtFmt = $ts ? date('d/m/Y H:i', $ts) : $dt;
                  } else {
                      $dtFmt = '';
                  }
                  $alvoTipo = $log['alvo_tipo'] ?? '';
                  $antes = !empty($log['dados_antes']) ? json_decode($log['dados_antes'], true) : null;
                  $depois = !empty($log['dados_depois']) ? json_decode($log['dados_depois'], true) : null;

                  $resumo = '';
                  if ($alvoTipo === 'plataforma' && is_array($depois)) {
                      $nome = $depois['nome'] ?? ($antes['nome'] ?? '');
                      $resumo = 'Plataforma ' . $nome;
                      if (isset($antes['taxa_percent'], $depois['taxa_percent'])) {
                          $resumo .= ' | taxa ' .
                              number_format((float)$antes['taxa_percent'], 2, ',', '.') . '% → ' .
                              number_format((float)$depois['taxa_percent'], 2, ',', '.') . '%';
                      }
                      if (isset($antes['custo_fixo'], $depois['custo_fixo'])) {
                          $resumo .= ' | custo fixo R$ ' .
                              number_format((float)$antes['custo_fixo'], 2, ',', '.') . ' → R$ ' .
                              number_format((float)$depois['custo_fixo'], 2, ',', '.');
                      }
                  } elseif ($alvoTipo === 'defaults' && is_array($depois)) {
                      $resumo = 'Padrões atualizados: ';
                      $camposInteresse = ['cmv','despesas_fixas','taxa_mentoria','custo_fixo_plataforma','taxa_plataforma','lucro_desejado'];
                      $partes = [];
                      foreach ($camposInteresse as $campo) {
                          if (isset($antes[$campo]) || isset($depois[$campo])) {
                              $antesV = $antes[$campo] ?? null;
                              $depoisV = $depois[$campo] ?? null;
                              if ($antesV !== $depoisV) {
                                  $partes[] = $campo . ' ' .
                                    number_format((float)$antesV, 2, ',', '.') . ' → ' .
                                    number_format((float)$depoisV, 2, ',', '.');
                              }
                          }
                      }
                      if ($partes) {
                          $resumo .= implode('; ', $partes);
                      }
                  }
                ?>
                <tr>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);"><?php echo htmlspecialchars($dtFmt); ?></td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);"><?php echo htmlspecialchars($log['alvo_tipo']); ?></td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);"><?php echo htmlspecialchars($log['alvo_user_email'] ?? '-'); ?></td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);"><?php echo htmlspecialchars($log['acao']); ?></td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);"><?php echo htmlspecialchars($log['referencia'] ?? ''); ?></td>
                  <td style="padding:4px 6px;border-bottom:1px solid var(--border-subtle);">
                    <?php echo htmlspecialchars($resumo); ?>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        <?php endif; ?>
      </div>
    </div>
  </div>
</div>

<?php include __DIR__ . '/footer.php'; ?>
