<?php
// admin_gateways.php
require __DIR__ . '/db.php';
require __DIR__ . '/functions.php';

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

if (empty($_SESSION['admin_autorizado'])) {
    header('Location: admin_login.php');
    exit;
}

$pageTitle = 'Admin - Gateways de Pagamento';

$mensagemSucesso = '';
$mensagemErro    = '';

$action = $_GET['action'] ?? '';
$id     = isset($_GET['id']) ? (int)$_GET['id'] : 0;

// SET DEFAULT via GET
if ($action === 'default' && $id > 0) {
    try {
        set_default_gateway($pdo, $id);
        $mensagemSucesso = 'Gateway definido como padrão com sucesso.';
    } catch (Throwable $e) {
        $mensagemErro = 'Erro ao definir gateway padrão: ' . $e->getMessage();
    }
}

// SALVAR (create/update)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_gateway'])) {
    $idPost     = (int)($_POST['id'] ?? 0);
    $name       = trim($_POST['name'] ?? '');
    $slug       = trim($_POST['slug'] ?? '');
    $provider   = trim($_POST['provider'] ?? '');
    $is_active  = isset($_POST['is_active']) ? 1 : 0;
    $is_default = isset($_POST['is_default']) ? 1 : 0;

    $cfg = [
        'access_token'    => trim($_POST['access_token'] ?? ''),
        'public_key'      => trim($_POST['public_key'] ?? ''),
        'base_url'        => trim($_POST['base_url'] ?? ''),
        'webhook_secret'  => trim($_POST['webhook_secret'] ?? ''),
        'checkout_url'    => trim($_POST['checkout_url'] ?? ''),
        'api_key'         => trim($_POST['api_key'] ?? ''),
        'encryption_key'  => trim($_POST['encryption_key'] ?? ''),
    ];
    foreach ($cfg as $k => $v) {
        if ($v === '') unset($cfg[$k]);
    }

    $config_json = json_encode($cfg, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

    if ($name === '' || $slug === '' || $provider === '') {
        $mensagemErro = 'Nome, slug e provider são obrigatórios.';
    } else {
        try {
            if ($idPost > 0) {
                $stmt = $pdo->prepare("
                    UPDATE payment_gateways
                    SET name = :name,
                        slug = :slug,
                        provider = :provider,
                        is_active = :is_active,
                        config_json = :config_json
                    WHERE id = :id
                ");
                $stmt->execute([
                    ':name'        => $name,
                    ':slug'        => $slug,
                    ':provider'    => $provider,
                    ':is_active'   => $is_active,
                    ':config_json' => $config_json,
                    ':id'          => $idPost,
                ]);

                if ($is_default) {
                    set_default_gateway($pdo, $idPost);
                }

                $mensagemSucesso = 'Gateway atualizado com sucesso.';

            } else {
                $stmt = $pdo->prepare("
                    INSERT INTO payment_gateways (name, slug, provider, is_active, is_default, config_json)
                    VALUES (:name, :slug, :provider, :is_active, 0, :config_json)
                ");
                $stmt->execute([
                    ':name'        => $name,
                    ':slug'        => $slug,
                    ':provider'    => $provider,
                    ':is_active'   => $is_active,
                    ':config_json' => $config_json,
                ]);
                $novoId = (int)$pdo->lastInsertId();

                if ($is_default && $novoId > 0) {
                    set_default_gateway($pdo, $novoId);
                }

                $mensagemSucesso = 'Gateway criado com sucesso.';
            }

        } catch (Throwable $e) {
            $mensagemErro = 'Erro ao salvar gateway: ' . $e->getMessage();
        }
    }
}

$gateways    = get_gateways($pdo);
$editGateway = null;
$action      = $_GET['action'] ?? '';
$id          = isset($_GET['id']) ? (int)$_GET['id'] : 0;

if ($action === 'edit' && $id > 0) {
    $editGateway = get_gateway_by_id($pdo, $id);
}

include __DIR__ . '/header.php';
?>

<div class="grid">
  <div class="card" style="grid-column: 1 / -1; max-width:980px; margin:0 auto;">
    <div class="card-header">
      <div class="card-title">Gateways de Pagamento</div>
      <span class="badge">ADMIN</span>
    </div>

    <?php if ($mensagemSucesso): ?>
      <div class="metric metric-highlight" style="margin-bottom:10px;">
        <div class="metric-label">
          <span>Sucesso</span>
        </div>
        <div class="metric-value metric-small">
          <?php echo htmlspecialchars($mensagemSucesso); ?>
        </div>
      </div>
    <?php endif; ?>

    <?php if ($mensagemErro): ?>
      <div class="metric metric-danger" style="margin-bottom:10px;">
        <div class="metric-label">
          <span>Erro</span>
        </div>
        <div class="metric-value metric-small">
          <?php echo htmlspecialchars($mensagemErro); ?>
        </div>
      </div>
    <?php endif; ?>

    <h3 style="font-size:0.9rem;margin-bottom:8px;">Gateways cadastrados</h3>

    <div class="field">
      <table class="admin-table">
        <thead>
          <tr>
            <th>ID</th>
            <th>Nome</th>
            <th>Slug</th>
            <th>Provider</th>
            <th>Ativo</th>
            <th>Padrão</th>
            <th>Ações</th>
          </tr>
        </thead>
        <tbody>
        <?php if ($gateways): ?>
          <?php foreach ($gateways as $gw): ?>
            <tr>
              <td><?php echo (int)$gw['id']; ?></td>
              <td><?php echo htmlspecialchars($gw['name']); ?></td>
              <td><?php echo htmlspecialchars($gw['slug']); ?></td>
              <td><?php echo htmlspecialchars($gw['provider']); ?></td>
              <td><?php echo $gw['is_active'] ? 'Sim' : 'Não'; ?></td>
              <td><?php echo $gw['is_default'] ? 'Sim' : '-'; ?></td>
              <td>
                <a href="admin_gateways.php?action=edit&id=<?php echo (int)$gw['id']; ?>" style="font-size:0.8rem;">Editar</a>
                <?php if (!$gw['is_default']): ?>
                  | <a href="admin_gateways.php?action=default&id=<?php echo (int)$gw['id']; ?>" style="font-size:0.8rem;">Definir padrão</a>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
        <?php else: ?>
          <tr><td colspan="7">Nenhum gateway cadastrado.</td></tr>
        <?php endif; ?>
        </tbody>
      </table>
      <div class="hint" style="margin-top:4px;">
        Configure credenciais e URLs e escolha qual gateway será o padrão do checkout.
      </div>
    </div>

    <hr class="divider">

    <h3 style="font-size:0.9rem;margin-bottom:8px;">
      <?php echo $editGateway ? 'Editar gateway' : 'Novo gateway'; ?>
    </h3>

    <?php
      $g   = $editGateway;
      $cfg = $g['config'] ?? [];
    ?>

    <form method="post" action="admin_gateways.php<?php echo $editGateway ? '?action=edit&id='.(int)$g['id'] : ''; ?>" style="margin:0;padding:0;">
      <input type="hidden" name="salvar_gateway" value="1">
      <input type="hidden" name="id" value="<?php echo $editGateway ? (int)$g['id'] : 0; ?>">

      <div class="field-row">
        <div class="field">
          <label>Nome</label>
          <input type="text" name="name" value="<?php echo htmlspecialchars($g['name'] ?? ''); ?>">
        </div>
        <div class="field">
          <label>Slug (identificador)</label>
          <input type="text" name="slug" value="<?php echo htmlspecialchars($g['slug'] ?? ''); ?>" placeholder="mercadopago, pagarme, asaas...">
        </div>
      </div>

      <div class="field-row">
        <div class="field">
          <label>Provider</label>
          <input type="text" name="provider" value="<?php echo htmlspecialchars($g['provider'] ?? ''); ?>" placeholder="mercadopago, pagarme, asaas...">
        </div>
        <div class="field">
          <label>Status</label>
          <label style="display:flex;align-items:center;gap:6px;">
            <input type="checkbox" name="is_active" <?php echo (!empty($g['is_active'])) ? 'checked' : ''; ?>>
            <span style="font-size:0.8rem;">Ativo</span>
          </label>
        </div>
      </div>

      <div class="field">
        <label>Definir como padrão?</label>
        <label style="display:flex;align-items:center;gap:6px;">
          <input type="checkbox" name="is_default" <?php echo (!empty($g['is_default'])) ? 'checked' : ''; ?>>
          <span style="font-size:0.8rem;">Usar este gateway como padrão no checkout</span>
        </label>
      </div>

      <hr class="divider">

      <div class="field">
        <label>Configurações específicas</label>
        <div class="hint">
          Mercado Pago: Access Token, Public Key, Base URL, Webhook Secret.<br>
          Pagar.me: API Key (usada na assinatura do postback) e/ou URL de checkout.<br>
          Asaas: URL de checkout e Webhook Secret (Authentication Token do painel Asaas).
        </div>
      </div>

      <div class="field-row">
        <div class="field">
          <label>Access Token</label>
          <input type="text" name="access_token" value="<?php echo htmlspecialchars($cfg['access_token'] ?? ''); ?>">
        </div>
        <div class="field">
          <label>Public Key</label>
          <input type="text" name="public_key" value="<?php echo htmlspecialchars($cfg['public_key'] ?? ''); ?>">
        </div>
      </div>

      <div class="field-row">
        <div class="field">
          <label>Base URL (app)</label>
          <input type="text" name="base_url" value="<?php echo htmlspecialchars($cfg['base_url'] ?? ''); ?>" placeholder="https://seu-dominio.com/app-preco">
        </div>
        <div class="field">
          <label>Webhook Secret / Auth Token</label>
          <input type="text" name="webhook_secret" value="<?php echo htmlspecialchars($cfg['webhook_secret'] ?? ''); ?>">
        </div>
      </div>

      <div class="field-row">
        <div class="field">
          <label>URL de checkout</label>
          <input type="text" name="checkout_url" value="<?php echo htmlspecialchars($cfg['checkout_url'] ?? ''); ?>">
        </div>
        <div class="field">
          <label>API Key</label>
          <input type="text" name="api_key" value="<?php echo htmlspecialchars($cfg['api_key'] ?? ''); ?>">
        </div>
      </div>

      <div class="field">
        <label>Encryption Key</label>
        <input type="text" name="encryption_key" value="<?php echo htmlspecialchars($cfg['encryption_key'] ?? ''); ?>">
      </div>

      <hr class="divider">

      <div class="field-row">
        <div class="field">
          <button type="submit" class="btn" style="justify-content:center;width:100%;">
            <span class="dot"></span> Salvar gateway
          </button>
        </div>
        <div class="field">
          <!-- Botão para voltar ao menu inicial do Admin -->
          <a href="admin.php" class="btn" style="justify-content:center;width:100%;text-decoration:none;">
            <span class="dot"></span> Voltar ao painel Admin
          </a>
        </div>
      </div>
    </form>
  </div>
</div>

<?php include __DIR__ . '/footer.php'; ?>
